<?php

namespace App\Jobs;

use App\Models\User;
use App\Models\RideDriver;
use Illuminate\Bus\Queueable;
use App\Events\AvailableRidesEvent;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Redis;
use App\Events\ClientRideRequestEvent;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Contracts\Queue\ShouldBeUnique;

class ExpandRideRadiusJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public $rideId;

     public function __construct($rideId)
    {
        $this->rideId = $rideId;
    }

    public function handle()
    {

        $prefix = config('database.redis.options.prefix') ?? '';
        $key = "{$prefix}ride_request:{$this->rideId}";

        if (!Redis::exists($key)) {
            return; // ride was accepted or removed
        }

        $data = json_decode(Redis::get($key), true);

        Log::info("Expanding radius for ride ID: {$this->rideId}", $data);
        // Expand the search radius
        
        $increment = env('RIDE_RADIUS_INCREMENT', 1);
        $data['radius'] += $increment;
        Log::info("Expanding radius for (after) ride ID: {$this->rideId}", $data);

        // Update Redis data
        // Redis::setex($key, 180, json_encode($data));


        Redis::del($key); 

        // Optionally wait a moment to ensure deletion is processed
        usleep(500000); // 0.5 seconds

        Redis::setex($key, 180, json_encode($data)); 



        $lat = $data['lat'];
        $lang = $data['lang'];
        $radius = $data['radius'];
        $vechile_type = $data['vehicle_type'] ?? null;
        $gender = $data['gender'] ?? null;

        // Fetch updated nearby drivers
        $query = User::query()
            ->where('is_online', 1)
            ->select('id')
            ->selectRaw('(
                6371 * acos(
                    cos(radians(?)) * cos(radians(lat)) *
                    cos(radians(lang) - radians(?)) +
                    sin(radians(?)) * sin(radians(lat))
                )
            ) AS distance', [$lat, $lang, $lat])
            ->having('distance', '<=', $radius)
            ->orderBy('distance');

        if ($vechile_type) {
            $query->where('vechile_type', $vechile_type);
        }

        if ($gender) {
            $query->where('gender', $gender);
        }

        $subscribersRealtime = $query->pluck('id');

        Log::info("Found nearby subscribersRealtime for ride ID: {$this->rideId}");
        Log::info("Found nearby subscribersRealtime ID: {$subscribersRealtime}");
        // Re-broadcast the ride to updated driver set
        broadcast(new AvailableRidesEvent([$data], $subscribersRealtime));
        
        self::dispatch($this->rideId)->delay(now()->addSeconds(30));
    }

}
