<?php

namespace App\Services;

use Carbon\Carbon;
use App\Models\User;
use App\Models\Client;
use App\Models\RideDriver;
use App\Events\RideRemovedEvent;
use App\Models\AppConfiguration;
use App\Jobs\ExpandRideRadiusJob;
use App\Models\RideDriverDetails;
use App\Traits\NotificationTrait;
use App\Events\AvailableRidesEvent;
use Illuminate\Support\Facades\Log;
use App\Repositories\RideRepository;
use App\Http\Resources\RidesResource;
use Illuminate\Support\Facades\Redis;
use App\Events\ClientRideRequestEvent;
use App\Services\CalculateDistanceService;

class RideService
{
    use NotificationTrait;
    private $rideRepository,$calculateDistanceService;
    public function __construct(RideRepository $rideRepository,CalculateDistanceService $calculateDistanceService)
    {
        $this->rideRepository = $rideRepository;  
        $this->calculateDistanceService = $calculateDistanceService;  
    }
    
    function rideDetails($request){
     
        $ride =RideDriver::find($request->id);
        return RidesResource::make($ride);
    }
    
    function getDriverRidesRates($request) {
        // Get the rates for the driver
        $currentDate = Carbon::now();
        $sixtyDaysAgo = $currentDate->subDays(60); 
        $endDate = Carbon::now()->addDays(1)->format('Y-m-d');
        $startDate = $sixtyDaysAgo->format('Y-m-d');
        $rates = RideDriver::query()
            ->whereUserId(auth()->id())
             ->where('updated_at','>=',$startDate)
            ->where('updated_at','<=',$endDate)
            ->select('id', 'car_rate', 'driver_rate', 'client_description', 'updated_at')
            ->orderByDESC("id")
            ->get();
    
        // Calculate the total and average using collection methods
        // $total_car_rate = $rates->sum('car_rate');
        // $total_driver_rate = $rates->sum('driver_rate');
        $avg_car_rate = $rates->avg('car_rate');
        $avg_driver_rate = $rates->avg('driver_rate');
    
        // Return the rates along with the totals and averages
        return [
            'rates' => $rates,
            // 'total_car_rate' => $total_car_rate,
            // 'total_driver_rate' => $total_driver_rate,
            'avg_car_rate' => number_format($avg_car_rate,2),
            'avg_driver_rate' =>number_format( $avg_driver_rate,2)
        ];
    }


    function startRide($request){
        Log::info("newRide " .json_encode($request->validated()) );
        $appConfiguration=AppConfiguration::whereCityId(auth()->user()->city_id)->first();
        $usedPrice=($request->vechile_type=='moto')?$appConfiguration->moto_km_price:$appConfiguration->km_price;
        $rideDriver['total_time']=$request->total_time??0;
        $rideDriver['total_distance']=$request->total_distance??0;
        $rideDriver['user_id']=auth()->id();
        $rideDriver['city_id']=auth()->user()->city_id;
        $rideDriver['start_fee']=$appConfiguration->start_fee;
        $rideDriver['km_fee']=$usedPrice;
        $rideDriver['waiting_fee']=$appConfiguration->waiting_price;
        // $rideDriver['actual_price']=$appConfiguration->start_fee;
        $rideDriver['total_price']=$request->total_distance *$usedPrice;
        $ride=RideDriver::create($rideDriver);
        $rideDriverDetails['ride_driver_id']=$ride->id;
        $keywords = '/\b(airport|مطار|airfield|terminal|aerodrome)\b/i';
        $isAirPort=false;

        // foreach(($request->points) as $point){
        foreach(json_decode($request->points,true) as $point){
            $rideDriverDetails['is_start_point']=0;
            $rideDriverDetails['is_end_point']=0;
            $rideDriverDetails['lat']=$point['lat'];
            $rideDriverDetails['lang']=$point['lang'];
            $rideDriverDetails['name']=$point['name'];
            $rideDriverDetails['fullName']=$point['fullName'];
            if ($rideDriverDetails['fullName']!=null && 
                preg_match($keywords, $rideDriverDetails['fullName'])) {
                   $isAirPort=true;
                   $airportPrice=$appConfiguration->airport_fee;
                   $ride->update(['airport_fee'=>$airportPrice]);
            }
            Log::info("order  id " . $point['order']);
            if($point['order'] == "1"){
                $rideDriverDetails['is_start_point']=1;
            }
            else{
                $rideDriverDetails['is_end_point']=1;
            }
            $this->rideRepository->createpoint($rideDriverDetails);
        }
        Log::info("newRide  id" .$rideDriverDetails['ride_driver_id'] );
        return $ride;
    }

    function clientStartRide($request){
        Log::info("newRide " .json_encode($request->validated()) );
        // $appConfiguration=AppConfiguration::whereCityId(auth()->user()->city_id)->first();
        $rideDriver['total_time']=$request->total_time??0;
        $rideDriver['total_distance']=$request->total_distance??0;
        $rideDriver['client_id']=auth()->guard("client")->user()->id;
        // $rideDriver['city_id']=auth()->user()->city_id;
        // $rideDriver['start_fee']=$appConfiguration->start_fee;
        // $rideDriver['km_fee']=$appConfiguration->km_price;
        // $rideDriver['waiting_fee']=$appConfiguration->waiting_price;
        // $rideDriver['actual_price']=$appConfiguration->start_fee;
        // $rideDriver['total_price']=$request->total_distance *$appConfiguration->km_price;
        $ride=RideDriver::create($rideDriver);
        $rideDriverDetails['ride_driver_id']=$ride->id;
        $keywords = '/\b(airport|مطار|airfield|terminal|aerodrome)\b/i';
        $isAirPort=false;
        // foreach(($request->points) as $point){
        
        foreach(json_decode($request->points,true) as $point){
            $rideDriverDetails['is_start_point']=0;
            $rideDriverDetails['is_end_point']=0;
            $rideDriverDetails['lat']=$point['lat'];
            $rideDriverDetails['lang']=$point['lang'];
            $rideDriverDetails['name']=$point['name'];
            $rideDriverDetails['fullName']=$point['fullName'];
            if ($rideDriverDetails['fullName']!=null && 
                preg_match($keywords, $rideDriverDetails['fullName'])) {
                   $isAirPort=true;
                   $airportPrice=$appConfiguration->airport_fee;
                   $ride->update(['airport_fee'=>$airportPrice]);
            }
            Log::info("order  id " . $point['order']);
            if($point['order'] == "1"){
                $rideDriverDetails['is_start_point']=1;
                $lat=$point['lat'];
                $lang=$point['lang'];
            }
            else{
                $rideDriverDetails['is_end_point']=1;
            }
            $this->rideRepository->createpoint($rideDriverDetails);
        }
        
        
        $notification=
        [
            'type' => "2",
            'title' => "رحلة جديدة",
            'title_en' => "New ride request",
            'message' => " طلب رحلة جديدة",
            'message_en' => "New ride request",
            "ride_id" =>$ride->id
        ];
        $subscribers=$this->getNearestDrivers($lat,$lang,5,$request->vechile_type,$request->gender);
        
        Log::info("client newRide  id" .$rideDriverDetails['ride_driver_id'] );
        Log::info("drivers " .json_encode($subscribers) );


        Log::info("ride before broadcastRideRequestToDrivers:" .$ride );
        // $this->broadcastRideRequestToDrivers($ride);
        
        
        $redisKey = "ride_request:{$ride->id}";

        $redisData = [
            'ride_id' => $ride->id,
            'ride'=> $ride,
            'rideDriverDetails'=> $rideDriverDetails,
            'lat' => $lat,
            'lang' => $lang,
            'start_time' => now()->toDateTimeString(),
            'vehicle_type' => $request->vechile_type,
            'gender' => $request->gender,
            'radius' => env('DEFAULT_RIDE_RADIUS', 1),
        ];
        

        Log::info("Saving ride to Redis", ['key' => $redisKey, 'data' => $redisData]);
        // Redis::setex($redisKey, 180, json_encode($redisData));
        // Redis::setex(config('database.redis.options.prefix') . "ride_request:{$ride->id}", 180, json_encode($redisData));
        $prefix = config('database.redis.options.prefix');
        $redisKey = "{$prefix}ride_request:{$ride->id}";
        Redis::connection()->client()->setex($redisKey, 180, json_encode($redisData));

        Log::info("Saved ride to Redis", ['key' => $redisKey]);

        // Use a small delay or wrap in a closure if needed
        sleep(1); // Optional, if Redis read is too fast
        

        $rides = $this->getAvailableRides();
        Log::info('Available rides fetched from Redis:', $rides);

        $subscribersRealtime = $this->getNearestDriversRealtime($lat, $lang, $redisData['radius'], $request->vechile_type, $request->gender);

        Log::info("Realtime subscribers: " . json_encode($subscribersRealtime));
        Log::info("Realtime redisData: " . json_encode($redisData));

        event(new AvailableRidesEvent([$redisData], $subscribersRealtime));


        // event(new AvailableRidesEvent($rides));
        
        $expandDelay = env('EXPAND_RIDE_RADIUS_DELAY', 30); // default to 30 seconds if not set
        ExpandRideRadiusJob::dispatch($ride->id)->delay(now()->addSeconds($expandDelay));
        $this->sendNotification($data_send=$notification,$subscribers);

        return $ride;
    }

    function getNearestDrivers($latitude, $longitude, $radius,$vechile_type,$gender) {
        
        $cond=["is_online"=>1];
        if($vechile_type){
            $cond['vechile_type']=$vechile_type;
        }
        if($gender){
            $cond['gender']=$gender;
        }
        $nearestDrivers = User::where(['is_online' => 1])
        ->select('device_id')
        ->selectRaw('( 6371 * acos( cos( radians(?) ) *
                       cos( radians( lat ) )
                       * cos( radians( lang ) - radians(?)
                       ) + sin( radians(?) ) *
                       sin( radians( lat ) ) )
                     ) AS distance', [$latitude, $longitude, $latitude])
        ->having('distance', '<=', $radius)
        ->orderBy('distance')
        ->pluck('device_id');
    
        return $nearestDrivers;
    }


    function editRideLocation($request,$status=false){
        Log::info("update Ride " .json_encode($request->all()) );

        // $point=($request->points); //json_decode($request->points,true);
        $point =json_decode($request->points,true);
        $rideDriverDetails['ride_driver_id']=$request->id;
        $rideDriverDetails['lat']=$point['lat'];
        $rideDriverDetails['lang']=$point['lang'];

        $lastPoint=RideDriverDetails::where('ride_driver_id',$request->id)
        ->orderByDESC('id')
        ->first(['lang','lat','id']);
        Log::info("lastPoint  " .$lastPoint);

        // $calculateDistanceResult=
        // $this->calculateDistanceService->calculateDistance($point['lang'],$point['lat'],$lastPoint->lang,$lastPoint->lat);
        Log::info('new lang '.$point['lang']);
        Log::info('new lat '.$point['lat']);
        Log::info('old lang '.$lastPoint->lang);
        Log::info('old lat '.$lastPoint->lat);
        // Log::info("calculateDistanceResult  id" .json_encode($calculateDistanceResult) );
        $distance= $request->distance;
        // $time= $calculateDistanceResult['time']/60000;

        $ride =RideDriver::find($request->id);
        // $appConfiguration=AppConfiguration::whereCityId(auth()->user()->city_id)->first();
        if($ride->user_id){
            User::whereId($ride->user_id)->update(['lang'=>$point['lang'],'lat'=>$point['lat']]);
        }
        if($ride->client_id){
            Client::whereId($ride->client_id)->update(['lang'=>$point['lang'],'lat'=>$point['lat']]);
        }
         if($status==true){
            // $km_price=$appConfiguration->km_price;
            $date1 = $ride->created_at;
               $date2 = now();

            $diff = $date2->diff($date1);
            $hoursDifference = $diff->h; // Hours
            $minutesDifference = $diff->i; // Minutes
            // Calculate the difference in minutes
            $date=($hoursDifference*60)+$minutesDifference;
        //   return $date;
            $updated=[
                'status'=>RideDriver::finsih,
                'actual_distance'=>$ride->actual_distance+$distance,
                'actual_price'=>($ride->actual_distance+$distance)*($ride->km_fee/1000),
                'total_waiting_time_fee'=>$request->speed * $ride->waiting_fee,
                'actual_time' =>$date
                // ,'total_waiting_time' =>$request->speed
            ];
        }else{
            $updated=[
                'actual_distance'=>$ride->actual_distance+$distance,
                'total_waiting_time_fee'=>$request->speed * $ride->waiting_fee,
                'total_waiting_time' =>$request->speed
            ];
        }
        
        $ride->update($updated);

        // Log::info("ride  " .$ride);
        Log::info("edit distance  " .$distance);
        // Log::info("edit time  " .$time);
        // Log::info("edit id  " .$request->id);

        $this->rideRepository->createpoint($rideDriverDetails);
        //  $ride->km_price=$appConfiguration->km_price;
        // $ride->waiting_price=$appConfiguration->waiting_price;
        $ride=$ride->load("user:id,name,phone,image");
        return RidesResource::make($ride);
        // return $ride;
    }

    function finishRide($request){
        $ride=$this->editRideLocation($request,true);
        return $ride;
    }

    function cancelRide($request){
        RideDriver::whereId($request->id)->update(["status"=>RideDriver::cancelled]);

        $prefix = config('database.redis.options.prefix') ?? '';
        $redisKey = "{$prefix}ride_request:{$request->id}";
        Redis::del($redisKey);

        Log::info("Ride cancelled and removed from Redis: {$redisKey}");
    }

    function initalAcceptRide($request){
        $ride =RideDriver::find($request->id);
        $data['status']=RideDriver::acceptNotDelivered;
        $data['user_id']=auth()->id();
        $appConfiguration=AppConfiguration::whereCityId(auth()->user()->city_id)->first();
        $usedPrice=(auth()->user()->vechile_type=='moto')?$appConfiguration->moto_km_price:$appConfiguration->km_price;
        $rideDriver['city_id']=auth()->user()->city_id;
        $rideDriver['start_fee']=$appConfiguration->start_fee;
        $rideDriver['km_fee']=$usedPrice;
        $rideDriver['waiting_fee']=$appConfiguration->waiting_price;
        $rideDriver['actual_price']=$appConfiguration->start_fee;
        $rideDriver['total_price']=$ride->total_distance *$usedPrice;
        $rideDetails=RideDriverDetails::where(['ride_driver_id'=>$request->id,'is_start_point'=>1])->first();
        $ride->update($data);
        $route['fLat']=auth()->user()->lat;
        $route['fLang']=auth()->user()->lang;
        $route['sLat']=$rideDetails->lat;
        $route['sLang']=$rideDetails->lang;
        $result=$this->graphHopperRoute($route);
        $client=Client::find($ride->client_id);
        $result['client_name']=$client->name;
        $result['client_phone']=$client->phone;

        $prefix = config('database.redis.options.prefix') ?? '';
        $redisKey = "{$prefix}ride_request:{$request->id}";
        Redis::del($redisKey);
        // event(new RideRemovedEvent($request->id));
        Log::info("Removed ride from Redis on driver accept: {$redisKey}");


        $prefix = config('database.redis.options.prefix') ?? '';
        $redisKey = "{$prefix}ride_request:{$request->id}";
        Redis::del($redisKey);
        Log::info("Removed ride from Redis on driver accept: {$redisKey}");

        return $result;
    }
    
    function graphHopperRoute($data){
        $fLat=$data['fLat'];
        $fLang=$data['fLang'];
        $sLat=$data['sLat'];
        $sLang=$data['sLang'];
        $curl = curl_init();
        curl_setopt_array($curl, array(
          CURLOPT_URL => "https://graphhopper.com/api/1/route?point=$fLat,$fLang&point=$sLat,$sLang&profile=car&locale=de&key=f57ce6e2-7333-45d7-9e25-80a0a8d63665&points_encoded=false",
          CURLOPT_RETURNTRANSFER => true,
          CURLOPT_ENCODING => '',
          CURLOPT_MAXREDIRS => 10,
          CURLOPT_TIMEOUT => 0,
          CURLOPT_FOLLOWLOCATION => true,
          CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
          CURLOPT_CUSTOMREQUEST => 'GET',
          CURLOPT_HTTPHEADER => array(
            'Accept: application/json'
          ),
        ));
        
        $response = curl_exec($curl);
        
        curl_close($curl);
        $response = json_decode($response, true);
        return ["distance"=>$response['paths'][0]['distance']/1000,
        "time"=>$response['paths'][0]['time']/60000];
    }
    
    
    function deliveredRide($request){
        $ride =RideDriver::find($request->id);
        $data['status']=RideDriver::acceptDelivered;
        $ride->update($data);
        $rideDetails=$ride->load('endPointsDetails');
        return  $rideDetails->endPointsDetails;
    }
    
     function customerArrivedRide($request){
        $ride =RideDriver::find($request->id);
        $data['status']=RideDriver::customerArrived;
        $ride->update($data);
        $rideDetails=$ride->load('endPointsDetails');
        return  $rideDetails->endPointsDetails;
    }


    public function broadcastRideRequestToDrivers($ride)
    {
        // Check if the ride has a start point
        Log::info("Broadcasting ride request for ride {$ride}");

        $startPoint = RideDriverDetails::where('ride_driver_id', $ride->id)
            ->where('is_start_point', 1)
            ->first(['lat', 'lang']);

        if (!$startPoint) {
            Log::warning("Start point not found for ride ID {$ride->id}");
            return;
        }

        $lat = $startPoint->lat;
        $lang = $startPoint->lang;
        $clientId = $ride->client_id;
        
        // event(new ClientRideRequestEvent($ride->id, $lat, $lang, $clientId));

        event(new ClientRideRequestEvent($ride));
    }

    public function getAvailableRides()
    {
        Log::info("Fetching available rides from Redis");
        
        
        $prefix = config('database.redis.options.prefix') ?? '';
        $keys = Redis::keys("{$prefix}ride_request:*");


        $rides = [];
        Log::info("Available rides keys: " . json_encode($keys));

        foreach ($keys as $key) {
            $value = Redis::get($key);
            if ($value) {
                $rides[] = json_decode($value, true);
            }
        }


        Log::info("Available rides: " . json_encode($rides));

        return $rides;
    }

    function getNearestDriversRealtime($latitude, $longitude, $radius = 1, $vechile_type = null, $gender = null)
    {
        $query = User::query()
            ->where('is_online', 1)
            ->select('id')
            ->selectRaw('(
                6371 * acos(
                    cos(radians(?)) * cos(radians(lat)) *
                    cos(radians(lang) - radians(?)) +
                    sin(radians(?)) * sin(radians(lat))
                )
            ) AS distance', [$latitude, $longitude, $latitude])
            ->having('distance', '<=', $radius)
            ->orderBy('distance');

        if ($vechile_type) {
            $query->where('vechile_type', $vechile_type);
        }

        if ($gender) {
            $query->where('gender', $gender);
        }

        return $query->pluck('id');
    }








}